module net.BurtonRadons.digc.digc;

import std.ctype;
import std.file;
import std.path;
import std.c.windows.windows;
import std.string;
import omflistexports;
import strip;
import libraryRegistrar;
import sys;
import std.c.stdio;
import net.BurtonRadons.dig.common.baseDirectory;
import net.BurtonRadons.digc.program;

extern (Windows) UINT GetSystemDirectoryA (LPTSTR lpBuffer, UINT uSize);

const char [] digver = `v0.0.14`;
char [] dmdexe = `\dmd\bin\dmd.exe`;
char [] libexe = `\dm\bin\lib.exe -p32`;
char [] dmd_bin = `\dmd\bin`;
char [] dmd_lib = `\dmd\lib`;
char [] dmc_lib = `\dm\lib`;
char [] dmd_src = `\dmd\src`;
char [] phobos = `\dmd\src\phobos`;
const char [] deffilename = "digc_dll.def";
const char [] sharedmainfilename = "digc_dll_main.d";
const char [] tempDirectory = "digc_temp";
char [] sharedLibraryInstallDir;

const uint libraryMarker = 0x55CF79B7;

const char [] dllmainbase = 
    "import windows;\n"
    "\n"
    "extern (C) void gc_init();\n"
    "extern (C) void gc_term();\n"
    "extern (C) void _minit();\n"
    "extern (C) void _moduleCtor();\n"
    "extern (C) void _moduleUnitTests();\n"
    "\n"
    "extern(Windows)\n"
    "BOOL DllMain(HINSTANCE hinst, ULONG reason, LPVOID reserved)\n"
    "{\n"
    "    switch (reason)\n"
    "    {\n"
    "        case DLL_PROCESS_ATTACH:\n"
    "            gc_init ();\n"
    "            _minit ();\n"
    "            _moduleCtor ();\n"
    "            _moduleUnitTests ();\n"
    "            break;\n"
    "\n"
    "        case DLL_PROCESS_DETACH:\n"
    "            gc_term ();\n"
    "            break;\n"
    "    }\n"
    "\n"
    "    return true;\n"
    "}";

/* It's important that the "1234" part at the top and everything before it not be changed. */
const char [] resbase =
    "\xFF\x18\x00\xFF\x01\x00\x30\x00"
    "1234<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\r\n"
    "<assembly xmlns=\"urn:schemas-microsoft-com:asm.v1\" manifestVersion=\"1.0\">\r\n"
    "<assemblyIdentity\r\n"
    "    version=\"1.0.0.0\"\r\n"
    "    processorArchitecture=\"X86\"\r\n"
    "    name=\"D.dig.%EXE%\"\r\n"
    "    type=\"win32\"\r\n"
    "/>\r\n"
    "<description>This is a dig program.</description>\r\n"
    "<dependency>\r\n"
    "    <dependentAssembly>\r\n"
    "        <assemblyIdentity\r\n"
    "            type=\"win32\"\r\n"
    "            name=\"Microsoft.Windows.Common-Controls\"\r\n"
    "            version=\"6.0.0.0\"\r\n"
    "            processorArchitecture=\"X86\"\r\n"
    "            publicKeyToken=\"6595b64144ccf1df\"\r\n"
    "            language=\"*\"\r\n"
    "        />\r\n"
    "    </dependentAssembly>\r\n"
    "</dependency>\r\n"
    "</assembly>";

const char [] resfilename = "digc_manifest.res";

extern (C) int system (char *args);

void showhelp (char [] program)
{
    printf (
"Dig Compiler %.*s\n"
"Usage:\n"
"  %.*s files.d ... { -switch }\n"
"\n"
"  files.d        D source files, possibly wildcards\n"
"  -not=wildcard  don't compile these files\n"
"  -exe=name      compile an executable (don't give an extension)\n"
"  -lib=name      compile a library (don't give an extension)\n"
"  -shared        create a shared library (a dll)\n"
"  /              begin a separate program\n"
"  -windowed      create a windowed executable rather than console\n"
"  -then=cmd      run command after successful compilation\n"
"  -install       copy created executables to the bin directory,\n"
"                 created libraries to the library directory,\n"
"                 and created shared libraries to the system directory.\n"
"\n"
"  -d             allow deprecated features\n"
"  -g             add symbolic debug info\n"
"  -gt            add trace profiling hooks\n"
"  -v             verbose\n"
"  -O             optimize\n"
"  -oobjdir       write .obj files to directory objdir\n"
"  -Llinkerflag   pass linkerflag to link\n"
"  -debug         compile in debug code\n"
"  -debug=level   compile in debug code <= level\n"
"  -debug=ident   compile in debug code identified by ident\n"
"  -inline        do function inlining\n"
"  -release       compile release version\n"
"  -unittest      compile in unit tests\n"
"  -version=level compile in version code >= level\n"
"  -version=ident compile in version code identified by ident\n"
"\n"
"  You can also expand variables using $(XXX), $[XXX], or ${XXX}:\n"
"      $(SystemDirectory)    Such as \"C:\\windows\\system32\"\n"
"      $(DMD_bin)            Such as \"\\dmd\\bin\"\n"
"      $(DMD_lib)            Such as \"\\dmd\\lib\"\n"
"      $(DMD_src)            Such as \"\\dmd\\src\"\n"
"      $(Phobos)             Such as \"\\dmd\\src\\phobos\"\n"
, digver, program);
}

bit winMain = false; /* WinMain mode, as opposed to console mode. */

bit [char []] versions;
int versionLevel = 0;
char [] result;

/** Return the number of seconds since 1970 since the file was last modified or zero if it doesn't exist. */
ulong fileTime (char [] filename)
{
    WIN32_FIND_DATA data;
    HANDLE handle;
    
    handle = FindFirstFileA (toStringz (std.string.replace (filename, "/", "\\")), &data);
    FindClose (handle);
    if (!handle)
        return 0;
    return data.ftLastWriteTime.dwLowDateTime | ((ulong) data.ftLastWriteTime.dwHighDateTime << 32);
}

bit endswith (char [] string, char [] suffix)
{
    return string.length >= suffix.length && string [string.length - suffix.length .. string.length] == suffix;
}

/* Quickly get string or single-symbol tokens from an array. */
char [] token (inout char *i, char *e)
{
    while (i < e)
    {
        if (i [0] == '/' && i [1] == '*')
        {
            i += 2;
            while (i < e)
            {
                if (i [0] == '*' && i [1] == '/')
                {
                    i += 2;
                    break;
                }
                i ++;
            }
        }
        else if (i [0] == '/' && i [1] == '/')
        {
            i += 2;
            while (i < e && *i != '\n')
                i ++;
        }
        else if (i [0] == '/' && i [1] == '+')
        {
            int depth = 1;

            i += 2;
            while (i < e)
            {
                if (i [0] == '/' && i [1] == '+')
                    depth ++, i += 2;
                else if (i [0] == '+' && i [1] == '/')
                {
                    i += 2;
                    depth --;
                    if (depth == 0)
                        break;
                }
                else
                    i ++;
            }
        }
        else if (isalpha (i [0]))
        {
            char *s = i;

            i ++;
            while (i < e && (isalnum (i [0]) || i [0] == '_'))
                i ++;

            return s [0 .. (int) (i - s)];
        }
        else if (isspace (i [0]))
            i ++;
        else
            return (i ++) [0 .. 1];
    }

    return null;
}

void addLibrary (inout char [] [] files, char [] value, char [] name)
{
    for (int c; ; c ++)
        if (c >= files.length)
        {
            //printf ("%.*s included %.*s\n", name, value);
            files ~= value;
            return;
        }
        else if (files [c] == value)
            return;
}

void skipContent (inout char *i, char *e)
{
    char [] t;

    while ((t = token (i, e)) !== null)
    {
        if (t == "}")
            return;
        if (t == "{")
            skipContent (i, e);
    }
}

void searchContent (inout char *i, char *e, inout char [] [] files, inout char [] moduleName)
{
    char [] t, name;

    while ((t = token (i, e)) !== null)
    {
        if (t == "}")
            return;
        if (t == "{")
        {
            searchContent (i, e, files, moduleName);
            continue;
        }

        if (t == "\"")
        {
            while (i < e && *i != '\"')
                if (*i == '\\')
                    i += 2;
                else
                    i ++;
            i ++;
            continue;
        }
        
        if (t == "\'")
        {
            while (i < e && *i != '\'')
                i ++;
            i ++;
            continue;
        }
        
        if (t == "version")
        {
        restart:
            t = token (i, e);
            if (t == "=")
            {
                t = token (i, e);
                versions [t] = true;
                continue;
            }

            if (t != "(")
                continue;

            if (token (i, e) in versions)
            {
                token (i, e); // skip the closing parenthesis.

                if (token (i, e) == "{") // read a block statement
                    searchContent (i, e, files, moduleName);
                else while ((t = token (i, e)) != ";" && t !== null)
                {
                    if (t == "{")
                    {
                        searchContent (i, e, files, moduleName);
                        break;
                    }
                }

                char *r = i;

                if (token (i, e) != "else")
                    i = r;
                else
                {
                    if (token (i, e) == "{")
                        skipContent (i, e);
                    else while ((t = token (i, e)) != ";" && t !== null)
                    {
                        if (t == "{")
                        {
                            skipContent (i, e);
                            break;
                        }
                    }
                }
            }
            else
            {
                skipContent (i, e);
                char *r = i;

                if (token (i, e) != "else")
                    i = r;
                else
                {
                    r = i;
                    if ((t = token (i, e)) == "{")
                        searchContent (i, e, files, moduleName);
                    else if (t == "version")
                        goto restart;
                    continue;
                }
            }
        }
        else if (t == "module")
        {
            moduleName = token (i, e);
            while ((t = token (i, e)) == ".")
                moduleName ~= "." ~ token (i, e);
        }
        else if (t == "import")
        {
            while ((t = token (i, e)) !== null)
            {
                name = t;
                while ((t = token (i, e)) == ".")
                    name ~= "." ~ token (i, e);

                Library lib = findImport (name);

                if (lib !== null && lib.name != result)
                {
                    addLibrary (files, lib.name, name);

                    for (int c; c < lib.deps.length; c ++)
                        addLibrary (files, lib.deps [c], "    ");
                }

                if (t == ";" || t != ",")
                    break;
            }
        }
    }
}

void searchFile (char [] filename, inout char [] [] files, inout char [] moduleName)
{
    char [] s, t, name;
    char *i, e;

    if (std.path.getExt (filename) != "d")
        return;
    s = (char []) std.file.read (filename);
    i = s;
    e = i + s.length - 1;

    searchContent (i, e, files, moduleName);
}

char [] dig_getDirName (char [] fullname)
{
    for (int i = fullname.length - 1; i >= 0; i --)
        if (fullname [i] == '/' || fullname [i] == '\\')
            return fullname [0 .. i];
        else if (fullname [i] == ':')
            return fullname [0 .. i + 1];
    return null;
}

char[] dig_getBaseName (char [] fullname)
    out (result)
    {
	assert(result.length <= fullname.length);
    }
    body
    {
	uint i;

	for (i = fullname.length; i > 0; i--)
	{
	    version(Win32)
	    {
		if (fullname[i - 1] == ':' || fullname[i - 1] == '\\')
		    break;
	    }
	    version(linux)
	    {
		if (fullname[i - 1] == '/')
		    break;
	    }
	}
	return fullname[i .. fullname.length];
    }

void variableExpand (inout char [] text, char [] name, char [] value)
{
    text = replace (text, "$(" ~ name ~ ")", value);
    text = replace (text, "$[" ~ name ~ "]", value);
    text = replace (text, "${" ~ name ~ "}", value);
    text = replace (text, "&(" ~ name ~ ")", value);
}

struct OSVERSIONINFO
{
  DWORD dwOSVersionInfoSize;
  DWORD dwMajorVersion;
  DWORD dwMinorVersion;
  DWORD dwBuildNumber;
  DWORD dwPlatformId;
  CHAR szCSDVersion[128];
}

extern (Windows) BOOL GetVersionExA (OSVERSIONINFO *);

void safeRemove (char [] name)
{
    DeleteFileA (toStringz (name));
}

bit readWildcard (Program program, char [] arg)
{
    WIN32_FIND_DATA data;
    HANDLE handle;

    arg = replace (arg, "/", "\\");

    if (std.string.find (arg, '?') < 0 && std.string.find (arg, '*') < 0)
    {
        program.files ~= arg;
        return false;
    }

    handle = FindFirstFileA (toStringz (arg), &data);
    if (handle == (HANDLE) INVALID_HANDLE_VALUE)
    {
        printf ("couldn't find files matching '%.*s'\n", arg);
        return true;
    }

    while (1)
    {
        char [] path;
        
        for (int d = arg.length - 1; d >= 0; d --)
            if (arg [d] == '\\')
            {
                path = arg [0 .. d + 1];
                break;
            }

        char [] filename = path ~ data.cFileName [0 .. strlen (data.cFileName)];

        program.files ~= filename.dup;
        if (!FindNextFileA (handle, &data))
            break;
    }

    FindClose (handle);
    return false;
}

int main (char [] [] argv)
{
    char [] [] aargs; /* Arguments passed to DMD */
    Program [] programs;
    Program program;
    char [] [] then; /* -then=command, execute command after successful compilation. */
    char [] [] thenProgram; /* -then-program=program, compile the indicated program after successful compilation. */
    int errors;
    bit verbose = true;
    bit verboseDesc = true;
    bit dontdeletetempdir = false;
    Package [] packages;

    dmdexe = digPlatformBaseDirectory ~ dmdexe;
    libexe = digPlatformBaseDirectory ~ libexe;
    dmd_bin = digPlatformBaseDirectory ~ dmd_bin;
    dmd_lib = digPlatformBaseDirectory ~ dmd_lib;
    dmd_src = digPlatformBaseDirectory ~ dmd_src;
    dmc_lib = digPlatformBaseDirectory ~ dmc_lib;
    phobos = digPlatformBaseDirectory ~ phobos;
    
    if (argv.length == 1)
    {
        showhelp (argv [0]);
        return 0;
    }

    loadLibraries ();

    /* Replace $(), $[], and ${} arguments. */
    {
        char [MAX_PATH * 2] SystemDirectoryS;
        char [] SystemDirectory;

        SystemDirectory = SystemDirectoryS [0 .. GetSystemDirectoryA (SystemDirectoryS, SystemDirectoryS.length)];

        for (int c; c < argv.length; c ++)
        {
            variableExpand (argv [c], "SystemDirectory", SystemDirectory);
            variableExpand (argv [c], "DMD_bin", dmd_bin);
            variableExpand (argv [c], "DMD_lib", dmd_lib);
            variableExpand (argv [c], "DMD_src", dmd_src);
            variableExpand (argv [c], "Phobos", phobos);
        }

        sharedLibraryInstallDir = SystemDirectory.dup;
    }

    program = new Program ();
    programs ~= program;

    for (int c = 1; c < argv.length; c ++)
    {
        char [] arg = argv [c];

        if (arg == "/")
        {
            if (program.name === null)
            {
                program.type = 'e';
                for (int c; c < program.files.length; c ++)
                    if (endswith (program.files [c], ".d"))
                    {
                        program.name = program.files [c];
                        break;
                    }
            }
            
            program = new Program ();
            programs ~= program;
        }
        else if (arg == "-dontdeletetempdir")
            dontdeletetempdir = true;
        else if (arg == "-windowed")
            winMain = true;
        else if (arg.length > 7 && arg [0 .. 7] == "-build=")
            packages ~= readBuildFile (arg [7 .. arg.length]);
        else if (arg.length > 14 && arg [0 .. 14] == "-then-program=")
            thenProgram ~= arg [14 .. arg.length];
        else if (arg.length > 6 && arg [0 .. 6] == "-then=")
            then ~= arg [6 .. arg.length];
        else if (arg == "-install")
            program.install = true;
        else if (arg == "-shared")
            program.shared = true;
        else if (arg.length > 5 && arg [0 .. 5] == "-lib=")
        {
            program.type = 'l';
            program.name = arg [5 .. arg.length];
        }
        else if (arg.length > 5 && arg [0 .. 5] == "-exe=")
        {
            program.type = 'e';
            program.name = arg [5 .. arg.length];
        }
        else if (arg.length > 5 && arg [0 .. 5] == "-not=")
        {
            WIN32_FIND_DATA data;
            HANDLE handle;

            handle = FindFirstFileA (arg [5 .. arg.length], &data);
            if (handle == (HANDLE) 0)
            {
                printf ("couldn't find files matching '%.*s'\n", arg), errors ++;
                continue;
            }

            for (int d = arg.length - 1; d >= 0; d --)
                if (arg [d] == '/')
                    arg [d] = '\\';

            while (1)
            {
                char [] path = arg [0 .. 5];
                
                for (int d = arg.length - 1; d >= 0; d --)
                    if (arg [d] == '\\')
                    {
                        path = arg [0 .. d + 1];
                        break;
                    }

                char [] filename = path [5 .. path.length] ~ data.cFileName [0 .. strlen (data.cFileName)];

                int sub = 0;
                for (int c = 0; c < program.files.length - sub; c ++)
                {
                    if (program.files [c] == filename)
                        sub ++;
                    program.files [c] = program.files [c + sub];
                }

                program.files = program.files [0 .. program.files.length - sub];
                if (!FindNextFileA (handle, &data))
                    break;
            }

            FindClose (handle);
        }
        else if (arg [0] == '-')
        {
            if (arg == "-c")
                program.doNotLink = true;
            aargs ~= arg;
        }
        else if (readWildcard (program, arg))
            errors ++;
    }

    if (program.name === null)
    {
        program.type = 'e';
        for (int c; c < program.files.length; c ++)
            if (endswith (program.files [c], ".d"))
            {
                program.name = program.files [c];
                break;
            }
    }
    
    if (errors)
        return 1;

    for (int d; d < programs.length; d ++)
    {
        int result;
        
        if ((result = programs [d].build (aargs, verbose, verboseDesc, dontdeletetempdir)) != 0)
        {
            writeLibraries ();
            return result;
        }
    }
    
    for (int d; d < packages.length; d ++)
    {
        int result;
        
        if ((result = packages [d].build (aargs, verbose, verboseDesc)) != 0)
        {
            writeLibraries ();
            return result;
        }
    }
    
    writeLibraries ();

    if (thenProgram.length)
        loadLibraries ();
        
    for (int d; d < thenProgram.length; d ++)
    {
        Program program;
        int result;
        
        for (int e; e < packages.length; e ++)
            if ((program = packages [e].findProgram (thenProgram [d])) !== null)
                break;
            
        if (!program)
        {
            printf ("Cannot find the program '%.*s'.", thenProgram [d]);
            return 1;
        }
        
        if ((result = program.build (aargs, verbose, verboseDesc, false)) != 0)
            return result;
        
        if (program.type == 'x')
        {
            program.run ();
            program.uninstall ();
        }
    }

    {
        int result = 0;

        for (int c; c < then.length; c ++)
        {
            if (verbose)
                printf("%.*s\n", then[c]);
            result = system (toStringz (then [c]));
            if (result < 0)
                return result;
        }

        return result;
    }
}
